define([
    'backbone',
    'Layout',

    'modules/new-appointment-request/views/direct/preferred-date/all-dates',
    'modules/new-appointment-request/views/direct/preferred-date/month-collapsible',

    'modules/new-appointment-request/resources/direct/preferred-date/time-slots-by-date-collection',
    'modules/new-appointment-request/resources/direct/preferred-date/time-slots-collection',
],
function(Backbone, Layout,
    AllDatesView,
    MonthCollapsibleView,
    TimeSlotsByDateCollection, TimeSlotsCollection) {
    'use strict';


    describe('All Dates View', function() {
        var view;
        var $view;
        var model;
        var timeSlotsByDateCollection;
        var layout;

        beforeEach(function() {
            timeSlotsByDateCollection = new TimeSlotsByDateCollection([
                {
                    date: '10/08/2015',
                    timeSlots: new TimeSlotsCollection(),
                },
                {
                    date: '10/09/2015',
                    timeSlots: new TimeSlotsCollection(),
                },
                {
                    date: '10/11/2015',
                    timeSlots: new TimeSlotsCollection([
                        {startDateTime: '10/11/2015 14:30:00'},
                        {startDateTime: '10/11/2015 15:00:00'},
                        {startDateTime: '10/11/2015 15:30:00'},
                    ]),
                },
                {
                    date: '10/13/2015',
                    timeSlots: new TimeSlotsCollection(),
                },
                {
                    date: '10/15/2015',
                    timeSlots: new TimeSlotsCollection(),
                },
                {
                    date: '11/11/2015',
                    timeSlots: new TimeSlotsCollection(),
                },
                {
                    date: '11/13/2015',
                    timeSlots: new TimeSlotsCollection(),
                },
                {
                    date: '11/15/2015',
                    timeSlots: new TimeSlotsCollection(),
                },
                {
                    date: '01/02/2016',
                    timeSlots: new TimeSlotsCollection(),
                },
            ]);
            layout = new Layout();
            layout.render();
        });

        afterEach(function() {
            view.destroy();
        });

        describe('on show', function() {
            beforeEach(function() {
                spyOn(AllDatesView.prototype, 'focusOnLegend').and.callThrough();
                spyOn(AllDatesView.prototype, 'viewNoDateTimeThatWorks');

                model = new Backbone.Model({desiredDate: '10/12/2015 00:00:00'});

                view = new AllDatesView({
                    model: model,
                    timeSlotsByDateCollection: timeSlotsByDateCollection,
                });

                layout.showChildView('content', view);
                $view = view.$el;
            });

            it('has a legend', function() {
                expect($view.find('legend').text())
                    .toEqual('*\xa0Select Date/Time:');
            });
            it('uses all-available-dates-container childViewContainer', function () {
                expect(view.childViewContainer).toBe('#all-available-dates-container');
            });
            it('has MonthCollapsibleView as its childView', function () {
                expect(view.childView).toBe(MonthCollapsibleView);
            });

            it('calls function to focusOnLegend()', function() {
                // unable to test focus in headless browser
                expect(view.focusOnLegend).toHaveBeenCalled();
                expect($.find('legend span[tabindex]').length).toEqual(1);
            });

            it("has a link `Don't see a date or time that works for you?`", function () {
                expect($view.find('#no-date-time-that-works').text()).toEqual('Don\'t see a date or time that works for you?');
            });

            it('calls viewNoDateTimeThatWorks on click of the link', function() {
                $view.find('#no-date-time-that-works').click();
                expect(view.viewNoDateTimeThatWorks).toHaveBeenCalled();
            });

        });
        describe('when the desired date is in the collection', function() {
            beforeEach(function() {
                spyOn(AllDatesView.prototype, 'focusOnLegend').and.callThrough();


                model = new Backbone.Model({desiredDate: '10/11/2015 00:00:00'});

                view = new AllDatesView({
                    model: model,
                    timeSlotsByDateCollection: timeSlotsByDateCollection,
                });

                layout.showChildView('content', view);
                $view = view.$el;
            });

            it('has a description', function() {
                expect($view.find('#available-dates-times-description').text())
                    .toEqual('Showing all available dates.');
            });

            it('expands the collapsible of the desired date', function() {
                var desiredDateCollapsible = $view.find('.available-dates .ui-collapsible:not(.ui-collapsible-collapsed)');
                expect(desiredDateCollapsible.length).toEqual(1);
                expect(desiredDateCollapsible.find('.ui-collapsible-heading span[aria-hidden="true"]').text()).toBe('10/11/2015');
            });
        });
        describe('when the desiredDate is not in the collection', function() {
            beforeEach(function() {
                spyOn(AllDatesView.prototype, 'focusOnLegend').and.callThrough();

                model = new Backbone.Model({desiredDate: '10/12/2015 00:00:00'});

                view = new AllDatesView({
                    model: model,
                    timeSlotsByDateCollection: timeSlotsByDateCollection,
                });

                layout.showChildView('content', view);
                $view = view.$el;
            });

            it('has a description', function() {
                expect($view.find('#available-dates-times-description').text())
                    .toEqual('No appointments are available on your preferred date. Showing all available dates.');
            });

            it('does not have any dates expanded', function() {
                var desiredDateCollapsible = $view.find('.available-dates .ui-collapsible:not(.ui-collapsible-collapsed)');
                expect(desiredDateCollapsible.length).toEqual(0);
            });
        });
    });
}
);
